#ifndef __CMidiTrack__
#define __CMidiTrack__

//	===========================================================================

#include <Collections/TCountedPointerArray.hpp>
#include <Basics/CSystemString.hpp>
#include "CMidiEvent.hpp"
#include "../Music/CTimeSignature.hpp"
using Exponent::Collections::TCountedPointerArray;
using Exponent::Basics::CSystemString;
using Exponent::Midi::CMidiEvent;
using Exponent::Music::CTimeSignature;

//	===========================================================================

namespace Exponent
{
	namespace Midi
	{
		/**
		 * @class CMidiTrack CMidiTrack.hpp
		 * @brief Provides an extensible array of midi events.\n
		 *
		 * Provides an extensible array of midi events. As this can cause dynamic allocation this class is not intended for real time usage when writing\n
		 * It should be noted that the delta times are stored as longs, as samples from the start of the track. Given that this value is in the range of \n
		 * 0 - 2147483647 this gives us a maximum track length of 13.51 *hours* assuming that you are running at a 44.1 sample rate. This value is obviously variable\n
		 * depending upon the samplerate, however, even running at 96khz you end up with at least 7 hours.. which should be enough for any musical work ;)
		 *
		 * @date 23/08/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * @note Documented online at <A HREF="http://www.borg.com/~jglatt/tech/midifile.htm">http://www.borg.com/~jglatt/tech/midifile.htm</A>
		 *
		 * $Id: CMidiTrack.hpp,v 1.3 2007/02/08 21:08:09 paul Exp $
		 */
		class CMidiTrack : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * Construction
			 */
			CMidiTrack();

			/**
			 * Destruction
			 */
			virtual ~CMidiTrack();

//	===========================================================================

			/**
			 * Add a track
			 * @param event The midi event to add
			 */
			void addEvent(CMidiEvent *event);

			/**
			 * Set the midi channel, events not on this midi channel are ignored
			 * @param midiChannel The midi channel to use
			 */
			void setMidiChannel(const long midiChannel) { m_midiChannel = midiChannel; }

			/**
			 * Set the track name
			 * @param name The name of the track
			 */
			void setTrackName(const CString &name) { m_trackName = name; }

			/**
			 * Get the midi channel
			 * @retval long The midi channel
			 */
			long getMidiChannel() const { return m_midiChannel; }

			/**
			 * Get the track name
			 * @retval const CString & The name of the track
			 */
			const CString &getTrackName() const { return m_trackName; }

			/**
			 * Get the number of events
			 * @retval long The number of events
			 */
			long getNumberOfEvents() const { return m_events.getArraySize(); }

			/**
			 * Get the event at the index
			 * @param index the index of the event you want
			 * @retval const CMidiEvent* The midi event or null on error
			 */
			const CMidiEvent *getEventAtIndex(const long index) const { return m_events.constElementAtIndex(index); }

			/**
			 * Get the event at the index
			 * @param index the index of the event you want
			 * @retval CMidiEvent* The midi event or null on error
			 */
			CMidiEvent *getMutableEventAtIndex(const long index) { return m_events.elementAtIndex(index); }

			/**
			 * Adjust event delta time by a ratio
			 * @param offset The offset ratio that you want to apply to all event delta positions
			 */
			void adjustDeltaTimeBy(const double offset);

//	===========================================================================

		protected:

//	===========================================================================
		
			TCountedPointerArray<CMidiEvent> m_events;			/**< events that we are dealing with */
			long m_midiChannel;									/**< The midi channel of this track */
			CString m_trackName;								/**< The name of the track */
		};
	}
}
#endif	// End of CMidiTrack.hpp